/* 
 * File:   DRV_MCP2515.h
 * Author: Wilhelm Leichtfried - C15279
 *
 * Created on July 1, 2015, 2:17 PM
 */
/*******************************************************************************
Copyright (c) 2014 released Microchip Technology Inc.  All rights reserved.

Microchip licenses to you the right to use, modify, copy and distribute
Software only when embedded on a Microchip microcontroller or digital signal
controller that is integrated into your product or third party product
(pursuant to the sublicense terms in the accompanying license agreement).

You should refer to the license agreement accompanying this Software for
additional information regarding your rights and obligations.

SOFTWARE AND DOCUMENTATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
EITHER EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF
MERCHANTABILITY, TITLE, NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE.
IN NO EVENT SHALL MICROCHIP OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER
CONTRACT, NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR
OTHER LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE OR
CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT OF
SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
(INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.
*******************************************************************************/

#include <stdint.h>

#ifndef DRV_MCP2515_H
#define	DRV_MCP2515_H

//#define DOXYGEN

#ifdef DOXYGEN
//! MCP2515 Driver defines
namespace MCP2515_Driver {
#endif

//! Number of TX Channels */
typedef enum {
    DRV_MCP2515_TX_CH0 = 0,
    DRV_MCP2515_TX_CH1,
    DRV_MCP2515_TX_CH2,
    DRV_MCP2515_TX_TOTAL_CHANNELS
} DRV_MCP2515_TX_CHANNEL;

//! Number of RX Channels */
typedef enum {
    DRV_MCP2515_RX_CH0 = 0,
    DRV_MCP2515_RX_CH1,
    DRV_MCP2515_RX_TOTAL_CHANNELS
} DRV_MCP2515_RX_CHANNEL;

//! Number of Filters
typedef enum {
    DRV_MCP2515_FILTER0 = 0,
    DRV_MCP2515_FILTER1,
    DRV_MCP2515_FILTER2,
    DRV_MCP2515_FILTER3,
    DRV_MCP2515_FILTER4,
    DRV_MCP2515_FILTER5,
    DRV_MCP2515_FILTER_TOTAL
} DRV_MCP2515_FILTER;

//! Number of Masks
typedef enum {
    DRV_MCP2515_MASK0 = 0,
    DRV_MCP2515_MASK1,
    DRV_MCP2515_MASK_TOTAL
} DRV_MCP2515_MASK;

//! Operation Modes */
typedef enum {
    DRV_MCP2515_NORMAL_MODE = 0x00,
    DRV_MCP2515_SLEEP_MODE = 0x01,
    DRV_MCP2515_LOOPBACK_MODE = 0x02,
    DRV_MCP2515_LISTEN_ONLY_MODE = 0x03,
    DRV_MCP2515_CONFIGURATION_MODE = 0x04,
    DRV_MCP2515_INVALID_MODE = 0xFF
} DRV_MCP2515_OPERATION_MODE;

//! Propagation Segment Time Quanta
typedef enum {
    DRV_MCP2515_PROP_1TQ = 0,
    DRV_MCP2515_PROP_2TQ,
    DRV_MCP2515_PROP_3TQ,
    DRV_MCP2515_PROP_4TQ,
    DRV_MCP2515_PROP_5TQ,
    DRV_MCP2515_PROP_6TQ,
    DRV_MCP2515_PROP_7TQ,
    DRV_MCP2515_PROP_8TQ
} DRV_MCP2515_PROPSEG_TQ;

//! Phase Segment1 Time Quanta
typedef enum {
    DRV_MCP2515_PHSEG1_1TQ = 0,
    DRV_MCP2515_PHSEG1_2TQ,
    DRV_MCP2515_PHSEG1_3TQ,
    DRV_MCP2515_PHSEG1_4TQ,
    DRV_MCP2515_PHSEG1_5TQ,
    DRV_MCP2515_PHSEG1_6TQ,
    DRV_MCP2515_PHSEG1_7TQ,
    DRV_MCP2515_PHSEG1_8TQ
} DRV_MCP2515_PHSEG1_TQ;

//! Phase Segment2 Time Quanta
typedef enum {
    DRV_MCP2515_PHSEG2_2TQ = 1,
    DRV_MCP2515_PHSEG2_3TQ,
    DRV_MCP2515_PHSEG2_4TQ,
    DRV_MCP2515_PHSEG2_5TQ,
    DRV_MCP2515_PHSEG2_6TQ,
    DRV_MCP2515_PHSEG2_7TQ,
    DRV_MCP2515_PHSEG2_8TQ
} DRV_MCP2515_PHSEG2_TQ;

//! Synchronization Jump Width
typedef enum {
    DRV_MCP2515_SJW_1TQ = 0,
    DRV_MCP2515_SJW_2TQ,
    DRV_MCP2515_SJW_3TQ,
    DRV_MCP2515_SJW_4TQ,
} DRV_MCP2515_SJW_TQ;

//! Sample Point Configuration
typedef enum {
    DRV_MCP2515_SAMPLE_ONCE = 0,
    DRV_MCP2515_SAMPLE_TRICE
} DRV_MCP2515_SAM_CONFIG;

//! Phase Seg2 Bit Time Length
typedef enum {
    DRV_MCP2515_PHSEG2_MIN = 0,
    DRV_MCP2515_PHSEG2_CNF3
} DRV_MCP2515_PHSEG2_CONFIG;

//! Start of Frame Output
typedef enum {
    DRV_MCP2515_CLKO_PIN = 0,
    DRV_MCP2515_SOF_PIN
} DRV_MCP2515_CLKO_SOF_CONFIG;

//! Wake Up Filter Enable
typedef enum {
    DRV_MCP2515_WAKFIL_DISABLE = 0,
    DRV_MCP2515_WAKFIL_ENABLE
} DRV_MCP2515_WAKFIL_CONFIG;

typedef struct _DRV_MCP2515_BIT_TIME_CONFIG {
    unsigned BRP : 6;
    unsigned PropSeg : 3;
    unsigned PhaseSeg1 : 3;
    unsigned PhaseSeg2 : 3;
    unsigned SJW : 2;
    unsigned BusSampling : 1;
    unsigned PhaseSeg2Config : 1;
    unsigned SOFPinConfig : 1;
    unsigned WakeUpFilterEnable : 1;
} DRV_MCP2515_BIT_TIME_CONFIG;

/* CAN Message Objects */
//! CAN Message Object ID
typedef struct _DRV_MCP2515_MSGOBJ_ID {
    unsigned SID:11;
    uint32_t EID:18;
} DRV_MCP2515_MSGOBJ_ID;

//! CAN TX Message Object Control
typedef struct _DRV_MCP2515_TX_MSGOBJ_CTRL {
    unsigned DLC:4;
    unsigned IDE:1;
    unsigned RTR:1;
} DRV_MCP2515_TX_MSGOBJ_CTRL;

//! CAN TX Message Object
typedef struct _DRV_MCP2515_TX_MSGOBJ {
    DRV_MCP2515_MSGOBJ_ID id;
    DRV_MCP2515_TX_MSGOBJ_CTRL ctrl;
} DRV_MCP2515_TX_MSGOBJ;

//! CAN RX Message Object Control
typedef struct _DRV_MCP2515_RX_MSGOBJ_CTRL {
    unsigned DLC:4;
    unsigned IDE:1;
    unsigned RTR:1;
    unsigned SRR:1;
    unsigned RB0:1;
    unsigned RB1:1;
} DRV_MCP2515_RX_MSGOBJ_CTRL;

//! CAN RX Message Object
typedef struct _DRV_MCP2515_RX_MSGOBJ {
    DRV_MCP2515_MSGOBJ_ID id;
    DRV_MCP2515_RX_MSGOBJ_CTRL ctrl;
} DRV_MCP2515_RX_MSGOBJ;

//! Read Status
typedef union _DRV_MCP2515_REG_BUFFER_STATUS {
    struct {
        unsigned RX0IF : 1;
        unsigned RX1IF : 1;
        unsigned TX0REQ : 1;
        unsigned TX0IF : 1;
        unsigned TX1REQ : 1;
        unsigned TX1IF : 1;
        unsigned TX2REQ : 1;
        unsigned TX3IF : 1;
    };
    unsigned char byte;
} DRV_MCP2515_REG_BUFFER_STATUS;

//! CAN Filter Object ID
typedef struct _DRV_MCP2515_FILTEROBJ_ID {
    unsigned SID:11;
    uint32_t EID:18;
    unsigned EXIDE:1;
} DRV_MCP2515_FILTEROBJ_ID;

//! CAN Mask Object ID
typedef struct _DRV_MCP2515_MASKOBJ_ID {
    unsigned MSID:11;
    uint32_t MEID:18;
} DRV_MCP2515_MASKOBJ_ID;

//! Receive Modes */
typedef enum {
    DRV_MCP2515_RXM_USE_FILTER = 0x0,
    DRV_MCP2515_RXM_RECEIVE_ALL = 0x3
} DRV_MCP2515_RXM_MODE;

//! Rollover to RXB1 if RXB0 is full
typedef enum {
    DRV_MCP2515_RXB_ROLLOVER_DISABLED = 0,
    DRV_MCP2515_RXB_ROLLOVER_ENABLED
} DRV_MCP2515_RXB_ROLLOVER;

//! Receive Mode configuration
typedef struct _DRV_MCP2515_RX_CONFIG {
    unsigned RXB0_ReceiveMode : 2;
    unsigned RXB1_ReceiveMode : 2;
    unsigned RXB_RollOver : 1;
} DRV_MCP2515_RX_CONFIG;

//! CAN Module Event (Interrupts); flags can be or'ed
typedef enum {
    DRV_MCP2515_NO_EVENT = 0,
    DRV_MCP2515_ALL_EVENTS = 0xFF,
    DRV_MCP2515_RXB0_EVENT = 0x01,
    DRV_MCP2515_RXB1_EVENT = 0x02,
    DRV_MCP2515_TXB0_EVENT = 0x04,
    DRV_MCP2515_TXB1_EVENT = 0x08,
    DRV_MCP2515_TXB2_EVENT = 0x10,
    DRV_MCP2515_ERROR_EVENT = 0x20,
    DRV_MCP2515_WAKEUP_EVENT = 0x40,
    DRV_MCP2515_MESSAGE_ERROR_EVENT = 0x80
} DRV_MCP2515_MODULE_EVENT;


#ifdef DOXYGEN
}

//! Driver MCP2515 API
namespace API_Functions {
#endif

/* API Functions */
//! Reset
void DRV_MCP2515_Reset();

//! Get Operation Mode
DRV_MCP2515_OPERATION_MODE DRV_MCP2515_OperationModeGet();

//! Select Operation Mode
void DRV_MCP2515_OperationModeSelect(DRV_MCP2515_OPERATION_MODE opMode);

//! Bit Time Configuration
void DRV_MCP2515_ConfigurationSet(DRV_MCP2515_BIT_TIME_CONFIG *config);

//! TX Channel Load
/*!
  Loads data into Transmit channel
  */
void DRV_MCP2515_TransmitChannelLoad(DRV_MCP2515_TX_CHANNEL channel, DRV_MCP2515_TX_MSGOBJ* txObj, const unsigned char *txd, unsigned char nBytes);

//! TX Channel Flush
/*!
  Set TXREG for a channel
  */
void DRV_MCP2515_TransmitChannelFlush(DRV_MCP2515_TX_CHANNEL channel);

//! Get Buffer Status
/*!
 Get Status of all buffers
  */
DRV_MCP2515_REG_BUFFER_STATUS DRV_MCP2515_BufferStatusGet();

//! Filter Configuration
/*!
  Configures ID of filter object
  */
void DRV_MCP2515_FilterConfigure(DRV_MCP2515_FILTER filter, DRV_MCP2515_FILTEROBJ_ID* fObj);

//! Mask Configuration
/*!
  Configures Mask of filter object
  */
void DRV_MCP2515_MaskConfigure(DRV_MCP2515_MASK mask, DRV_MCP2515_MASKOBJ_ID* mObj);

//! Get Received Message
/*!
  Reads Received message from channel
  */
void DRV_MCP2515_ReceiveMessageGet(DRV_MCP2515_RX_CHANNEL channel, DRV_MCP2515_RX_MSGOBJ* rxObj, unsigned char *rxd, unsigned char nBytes);

//! Set Receive Configuration
/*!
 Sets Receive mode, and roll over from RXB0 to RXB1
 */
void DRV_MCP2515_ReceiveConfigurationSet(DRV_MCP2515_RX_CONFIG* config);

//! Module Event Get
/*!
  Reads interrupt Flags
  */
DRV_MCP2515_MODULE_EVENT DRV_MCP2515_ModuleEventGet();

//! Module Event Clear
/*!
  Clears interrupt Flags
  */
void DRV_MCP2515_ModuleEventClear(DRV_MCP2515_MODULE_EVENT flags);

//! Module Event Enable
/*!
  Enables interrupts
  */
void DRV_MCP2515_ModuleEventEnable(DRV_MCP2515_MODULE_EVENT flags);

//! Module Event Disable
/*!
  Disables interrupts
  */
void DRV_MCP2515_ModuleEventDisable(DRV_MCP2515_MODULE_EVENT flags);

/* Register Access Functions */

//! Read Register
unsigned char DRV_MCP2515_RegisterRead(unsigned char address);

//! Write Register
void DRV_MCP2515_RegisterWrite(unsigned char address, unsigned char data);

//! Bit Modify Register
void DRV_MCP2515_RegisterBitModify(unsigned char address, unsigned char mask, unsigned char data);

//! Read Buffer
void DRV_MCP2515_BufferRead(unsigned char address, unsigned char *data, unsigned char nBytes);

//! Write Buffer
void DRV_MCP2515_BufferWrite(unsigned char address, unsigned char *data, unsigned char nBytes);

#ifdef DOXYGEN
}
#endif

#ifdef DOXYGEN
//! MCP2515 Register defines
namespace MCP2515_Registers {
#endif

/* Register bit fields */
//! CANCTRL
typedef union {
    struct {
        unsigned ClockOutPrescaler : 2;
        unsigned ClockOutEnable : 1;
        unsigned OneShotMode : 1;
        unsigned TxAbortAll : 1;
        unsigned OpModeRequest : 3;
    };
    unsigned char byte;
} DRV_MCP2515_REG_CANCTRL;

//! CANSTAT
typedef union {
    struct {
        unsigned Unimplemented0 : 1;
        unsigned InterruptCode : 3;
        unsigned Unimplemented1 : 1;
        unsigned OpMode : 3;
    };
    unsigned char byte;
} DRV_MCP2515_REG_CANSTAT;

//! CNF1
typedef union {
    struct {
        unsigned BRP : 6;
        unsigned SJW : 2;
    };
    unsigned char byte;
} DRV_MCP2515_REG_CNF1;

//! CNF2
typedef union {
    struct {
        unsigned PropSeg : 3;
        unsigned PhaseSeg1 : 3;
        unsigned SAM : 1;
        unsigned BTLMode : 1;
    };
    unsigned char byte;
} DRV_MCP2515_REG_CNF2;

//! CNF3
typedef union {
    struct {
        unsigned PhaseSeg2 : 3;
        unsigned Unimplemented0 : 3;
        unsigned WakeUpFilterEnable : 1;
        unsigned SOF : 1;
    };
    unsigned char byte;
} DRV_MCP2515_REG_CNF3;

//! TXBNCTRL
typedef union {
    struct {
        unsigned TxPriority : 2;
        unsigned Unimplemented0 : 1;
        unsigned TxRequest : 1;
        unsigned TxError : 1;
        unsigned ArbitrationLost : 1;
        unsigned MessageAborted : 1;
        unsigned Unimplemented1 : 1;
    };
    unsigned char byte;
} DRV_MCP2515_REG_TXBNCTRL;

//! TXRTSCTRL
typedef union {
    struct {
        unsigned B0RTSM : 1;
        unsigned B1RTSM : 1;
        unsigned B2RTSM : 1;
        unsigned B0RTS : 1;
        unsigned B1RTS : 1;
        unsigned B2RTS : 1;
        unsigned Unimplemented0 : 2;
    };
    unsigned char byte;
} DRV_MCP2515_REG_TXRTSCTRL;

//! TXBnSIDL
typedef union {
    struct {
        unsigned EIDHH : 2;
        unsigned Unimplemented0 : 1;
        unsigned IDE : 1;
        unsigned Unimplemented1 : 1;
        unsigned SIDL : 3;
    };
    unsigned char byte;
} DRV_MCP2515_REG_TXBnSIDL;

//! TXBnDLC
typedef union {
    struct {
        unsigned DLC : 4;
        unsigned Unimplemented0 : 2;
        unsigned RTR : 1;
        unsigned Unimplemented1 : 1;
    };
    unsigned char byte;
} DRV_MCP2515_REG_TXBnDLC;

//! RXBnSIDL
typedef union {
    struct {
        unsigned EIDHH : 2;
        unsigned Unimplemented0 : 1;
        unsigned IDE : 1;
        unsigned SRR : 1;
        unsigned SIDL : 3;
    };
    unsigned char byte;
} DRV_MCP2515_REG_RXBnSIDL;

//! RXBnDLC
typedef union {
    struct {
        unsigned DLC : 4;
        unsigned RB0 : 1;
        unsigned RB1 : 1;
        unsigned RTR : 1;
        unsigned Unimplemented1 : 1;
    };
    unsigned char byte;
} DRV_MCP2515_REG_RXBnDLC;

//! RXB0CTRL
typedef union {
    struct {
        unsigned FILTHIT : 1;
        unsigned Unimplemented1 : 1;
        unsigned BUKT : 1;
        unsigned RXRTR : 1;
        unsigned Unimplemented2 : 1;
        unsigned RXM :2;
        unsigned Unimplemented3 : 1;
    };
    unsigned char byte;
} DRV_MCP2515_REG_RXB0CTRL;

///! RXB1CTRL
typedef union {
    struct {
        unsigned FILTHIT : 3;
        unsigned RXRTR : 1;
        unsigned Unimplemented1 : 1;
        unsigned RXM :2;
        unsigned Unimplemented3 : 1;
    };
    unsigned char byte;
} DRV_MCP2515_REG_RXB1CTRL;

/* Register addresses */
#define cREGADDR_CANCTRL	0x0F
#define cREGADDR_CANSTAT	0x0E

#define cREGADDR_CNF1   	0x2A
#define cREGADDR_CNF2   	0x29
#define cREGADDR_CNF3    	0x28

#define cREGADDR_TXB0CTRL	0x30
#define cREGADDR_TXB0ID 	0x31
#define cREGADDR_TXB0DATA 	0x36

#define cREGADDR_TXB1CTRL	0x40
#define cREGADDR_TXB1ID  	0x41
#define cREGADDR_TXB1DATA 	0x46

#define cREGADDR_TXB2CTRL	0x50
#define cREGADDR_TXB2ID 	0x51
#define cREGADDR_TXB2DATA 	0x56

#define cREGADDR_RXF0           0x00
#define cREGADDR_RXF1           0x04
#define cREGADDR_RXF2           0x08
#define cREGADDR_RXF3           0x10
#define cREGADDR_RXF4           0x14
#define cREGADDR_RXF5           0x18

#define cREGADDR_RXM0           0x20
#define cREGADDR_RXM1           0x24

#define cREGADDR_RXB0CTRL	0x60
#define cREGADDR_RXB0ID 	0x61
#define cREGADDR_RXB0DATA 	0x66

#define cREGADDR_RXB1CTRL	0x70
#define cREGADDR_RXB1ID 	0x71
#define cREGADDR_RXB1DATA 	0x76

#define cREGADDR_CANINTE	0x2B
#define cREGADDR_CANINTF	0x2C

#ifdef DOXYGEN
}
#endif

#endif	/* DRV_MCP2515_H */

